import os
import json
import requests
import logging
from datetime import datetime

 

# ==========================
# CONFIGURACIÓN DE LOGS
# ==========================
# Definimos ruta del log (en la misma carpeta del script)
LOG_FILE = os.path.join(os.getcwd(), "pedido_log.txt")

try:
    # Validamos que se pueda crear/escribir
    with open(LOG_FILE, "a", encoding="utf-8") as f:
        f.write("=== Inicio de ejecución ===\n")

    logging.basicConfig(
        level=logging.INFO,
        format="%(asctime)s [%(levelname)s] %(message)s",
        handlers=[
            logging.FileHandler(LOG_FILE, mode="a", encoding="utf-8")
        ]
    )

    # Avisamos al usuario dónde se va a guardar el log
    print(f"📝 Todos los pasos se registrarán en el archivo de log: {LOG_FILE}")
    logging.info("Archivo de log creado correctamente.")

except Exception as e:
    print(f"❌ No se pudo crear el archivo de log: {LOG_FILE}")
    print(f"Error: {e}")
    exit(1)


# ==========================
# CONFIGURACIONES
# ==========================
CHANNELS = {
    "tnannastein": ["pickup", "delivery"],
    "magento": ["pickup", "delivery"],
    "MELI": ["pickup", "delivery"],
    "farmaciadigital": ["pickup", "delivery"],
    "farmatouch": ["pickup", "delivery"],
    "RAPPI": ["delivery"],
    "PEYA": ["delivery"],
    "benvida": ["pickup", "delivery"],
    "vtex": ["delivery"]
}

EXTERNAL_FILE = "external_number.txt"
INITIAL_EXTERNAL = 40001400
SHIPPING_COST = 15
PROMO_DISCOUNT = 2400

URLS_CLIENTES = {
    "Farmacia Central Oeste": "https://bmc-foco.dev.napse.global:8444/order/create",
    "Proyecto 2": "https://dev.3napse.global:8444/order/create",
    "Proyecto 3": "https://napse.4global:8444/order/create"
}

# ==========================
# SELECCIÓN CLIENTE
# ==========================
print("Seleccione el cliente:")
for i, cliente in enumerate(URLS_CLIENTES.keys(), start=1):
    print(f"{i}. {cliente}")

opcion = input("Ingrese el número de la opción: ").strip()
try:
    opcion = int(opcion)
    clientes = list(URLS_CLIENTES.keys())
    if 1 <= opcion <= len(clientes):
        cliente_seleccionado = clientes[opcion - 1]
        URL = URLS_CLIENTES[cliente_seleccionado]
        logging.info(f"Cliente seleccionado: {cliente_seleccionado}")
        logging.info(f"Se utilizará la URL: {URL}")
    else:
        logging.error("Opción inválida.")
        exit(1)
except ValueError:
    logging.error("Debe ingresar un número.")
    exit(1)

# ==========================
# FUNCIONES
# ==========================
def get_next_external_number():
    logging.info("Obteniendo siguiente externalNumber...")
    if os.path.exists(EXTERNAL_FILE):
        with open(EXTERNAL_FILE, "r") as f:
            num = int(f.read().strip())
    else:
        num = INITIAL_EXTERNAL - 1

    next_num = num + 1
    with open(EXTERNAL_FILE, "w") as f:
        f.write(str(next_num))
    logging.info(f"ExternalNumber generado: {next_num}")
    return str(next_num)

# ==========================
# Variables
# ==========================
item1 = "1477"
item2 = "9018"
store1 = "100000001"
store2 = "100000002"
store3 = "100000003"
store4 = "100000004"

# ==========================
# INTERACCIÓN CON EL USUARIO
# ==========================
print("Seleccione un canal:")
for i, (ch, modes) in enumerate(CHANNELS.items(), 1):
    marca = " *P/D" if len(modes) == 2 else ""
    print(f"{i}. {ch}{marca}")

canal_idx = int(input("Número de canal: ").strip()) - 1
channelCode = list(CHANNELS.keys())[canal_idx]
delivery_modes = CHANNELS[channelCode]
logging.info(f"Canal seleccionado: {channelCode}")

# Selección de entrega
if delivery_modes == ["delivery"]:
    deliveryOrPickup = "delivery"
    logging.info(f"Canal {channelCode} solo acepta delivery. Asignado automáticamente.")
else:
    while True:
        print("\nTipo de entrega:")
        print("1. pickup")
        print("2. delivery")
        tipo_entrega_input = input("Seleccione 1 o 2: ").strip()
        if tipo_entrega_input == "1":
            deliveryOrPickup = "pickup"
            break
        elif tipo_entrega_input == "2":
            deliveryOrPickup = "delivery"
            break
        else:
            print("❌ Opción inválida.")
logging.info(f"Tipo de entrega seleccionado: {deliveryOrPickup}")

# Selección de storeCode
print("\n3. Indicar Tienda")
print("1. 100000001")
print("2. 100000002")
print("3. 100000003")
print("4. 100000004")
store_input = input("Seleccione 1 al 4: ").strip()
if store_input == "1":
    storeCode = store1
elif store_input == "2":
    storeCode = store2
elif store_input == "3":
    storeCode = store3
elif store_input == "4":
    storeCode = store4
else:
    logging.error("Opción inválida para tienda. Debe elegir entre 1 y 4.")
    exit(1)
logging.info(f"Tienda seleccionada: {storeCode}")

# ¿Está pagado?
isPaid = input("¿Está pagado? (s/n): ").strip().lower() == "s"
logging.info(f"Pedido pagado: {isPaid}")

# ¿Aplica promo?
aplicaPromo = input("¿Quiere aplicar descuento de PROMO? (s/n): ").strip().lower() == "s"
logging.info(f"Aplicar PROMO: {aplicaPromo}")

# ¿Aplica descuento manual?
aplica_descuento_manual = input("¿Quiere aplicar un descuento manual? (s/n): ").strip().lower() == "s"
logging.info(f"Aplicar descuento manual: {aplica_descuento_manual}")

# ==========================
# ARMADO DEL PEDIDO
# ==========================
logging.info("Armando pedido...")
creationDate = datetime.now().strftime("%Y-%m-%d")
externalNumber = get_next_external_number()

pedido = {
    "externalNumber": externalNumber,
    "creationDate": creationDate,
    "channelCode": channelCode,
    "currencyCode": "ARS",
    "manualDiscountAmount": 0,
    "netAmount": 11000,
    "orderTypeCode": "web",
    "isPaid": isPaid,
    "party": {
        "code": "",
        "firstName": "Alejandro",
        "lastName": "Isola",
        "email": "Leo@napse.global",
        "typeCode": "Person",
        "identificationType": 1,
        "identifier": "2995028",
        "partyContactMethods": [
            {
                "name": "personal",
                "address": {
                    "firstLine": "Av San Aurelio 7844",
                    "countryCode": "AR",
                    "stateCode": "BA",
                    "cityCode": "BSAS_194"
                },
                "emailAddress": "alejandro.silva@napse.global",
                "telephone": {
                    "telephoneNumber": "0342 4744900"
                },
            }
        ]
    },
    "detail": [
        {
            "itemCode": item1,
            "description": " ",
            "storeCode": storeCode,
            "locationCode": "DEP1",
            "unitPrice": 5000,
            "orderedQty": 2,
            "deliveryOrPickup": deliveryOrPickup,
            "deliveryCompany": "100000003",
            "pickupLocationCode": "100000003",
            "packageId": "0",
            "partyContactMethodName": "personal",
            "pickupInformation": "Info: Alejandro Silva Mirgor retira por tienda."
        },
        {
            "itemCode": item2,
            "description": "PBRAVA DESOD. LUBRICANTE-sachet cajax20 cód.12060",
            "storeCode": storeCode,
            "locationCode": "DEP1",
            "unitPrice": 1000,
            "orderedQty": 1,
            "deliveryOrPickup": deliveryOrPickup,
            "deliveryCompany": "100000003",
            "pickupLocationCode": "100000003",
            "packageId": "0",
            "partyContactMethodName": "personal",
            "pickupInformation": "Info: Alejandro Silva Mirgor retira por tienda."
        }
    ]
}

# Agregar envío si corresponde
if deliveryOrPickup == "delivery":
    logging.info("Agregando costo de envío al pedido...")
    envio_item = {
        "itemCode": "ENVIO",
        "description": "Costo de Envio",
        "storeCode": storeCode,
        "locationCode": "DEP1",
        "unitPrice": SHIPPING_COST,
        "orderedQty": 1,
        "deliveryOrPickup": "delivery",
        "deliveryCompany": "magento",
        "pickupLocationCode": "100000003",
        "packageId": "0",
        "partyContactMethodName": "personal",
        "pickupInformation": "Info: Alejandro Silva retira por tienda.",
        "notes": "RECETA=true|CODIGO=999999|MATMED=88888888888888",
        "sellerID": "omni",
        "sellerName": "Usuario para WF OMNI"
    }
    pedido["detail"].append(envio_item)
    pedido["netAmount"] += SHIPPING_COST

# Descuento manual
if aplica_descuento_manual:
    try:
        descuento_manual = float(input("Ingrese el importe del descuento manual: ").strip())
        if descuento_manual > 0 and descuento_manual <= pedido["netAmount"]:
            pedido["manualDiscountAmount"] = descuento_manual
            pedido["netAmount"] -= descuento_manual
            logging.info(f"Descuento manual aplicado: {descuento_manual}")
        else:
            logging.warning("Importe de descuento inválido.")
    except ValueError:
        logging.warning("Valor ingresado no es numérico para descuento manual.")

# Promo
if aplicaPromo:
    pedido["detail"][0]["priceModifiers"] = [
        {"percent": 20, "amount": PROMO_DISCOUNT, "reasonText": "38"}
    ]
    pedido["netAmount"] -= PROMO_DISCOUNT
    logging.info(f"Descuento PROMO aplicado: {PROMO_DISCOUNT}")

# Payments
if isPaid:
    pago = {
        "tender": "EL",
        "amount": pedido["netAmount"],
        "currencyCode": "ARS",
        "planDescriptor": "12",
        "sellerName": "vISA CREDITO",
        "authorizationCode": "9187840193",
        "couponNumber": "123",
        "lotNumber": "4321",
        "installments": 6,
        "referenceNumber": "383409####0415"
    }
    pedido["payments"] = [pago]
else:
    pedido["payments"] = []

# ==========================
# MOSTRAR PEDIDO PARA REVISAR
# ==========================
print("\n📝 Pedido generado (vista previa):\n")
print(json.dumps(pedido, indent=4, ensure_ascii=False))
print(f"\n💲 Monto final netAmount: {pedido['netAmount']}")
logging.info("Pedido generado y mostrado para revisión.")
logging.info(json.dumps(pedido, indent=4, ensure_ascii=False))
confirm = input("\n¿Desea enviar este pedido? (s/n): ").strip().lower()
if confirm not in ["s", "si"]:
    logging.warning("Envío cancelado por el usuario.")
    exit(0)

# ==========================
# ENVÍO
# ==========================
token = input("\nIngrese el x-access-token: ").strip()
headers = {"Content-Type": "application/json", "x-access-token": token}

try:
    logging.info("Enviando pedido al endpoint...")
    response = requests.post(URL, headers=headers, json=pedido)
    logging.info(f"Enviado a: {URL}")
    logging.info(f"Status: {response.status_code}")
    logging.info(f"Respuesta: {response.text}")
    print(f"\n📡 Enviado a: {URL}")
    print("Status:", response.status_code)
    print("Respuesta:", response.text)
except Exception as e:
    logging.error(f"Error al enviar: {e}")
    print("❌ Error al enviar:", str(e))

